import { RumSiteVisitsResponse } from '../zod/rum-site-visits.schema';
import { OtpLoggingResponse } from '../zod/otp-logging.schema'
import { RumSessionTimeResponse } from '../zod/rum-session-time.schema';

export function transformOtpLoggingResponse(response: OtpLoggingResponse) {
  return response.map(
    (r: OtpLoggingResponse[number]) => ({
      from: parseCoordinates(r.from),
      to: parseCoordinates(r.to),
    })
  )
}

const parseCoordinates = (coordinates: string): [number, number] => {
  const match = coordinates.match(/^\(([-\d.]+),\s*([-\d.]+)\)$/);

  if (!match) {
    throw new Error("Invalid coordinate format. Expected '(x,y)'");
  }

  const x = parseFloat(match[1]);
  const y = parseFloat(match[2]);

  return [x, y];
};

export function transformRumMonthlyVisitsResponse(response: RumSiteVisitsResponse) {
  return Object.entries(response).map(([date, value]) => {
    const parsedDate = new Date(date);
    const month = parsedDate.toLocaleDateString('en-US', { month: '2-digit' })
    const year = parsedDate.toLocaleDateString('en-US', { year: '2-digit' })
    const isJanuary = month == '01' ? true : false;
    return {
      label: parsedDate.toLocaleDateString('en-US', { month: 'short' }) + (isJanuary ? ` '${year}` : ''),
      month,
      year,
      visits: Number(value),
    };
  }).reverse();
}

export function transformRumDailyVisitsResponse(response: RumSiteVisitsResponse) {
  return Object.entries(response).map(([date, value]) => {
    const parsedDate = new Date(date);
    const day = parsedDate.toLocaleDateString('en-US', { day: '2-digit' })
    const month = parsedDate.toLocaleDateString('en-US', { month: '2-digit' })
    return {
      label: `${month}/${day}`,
      day,
      month,
      visits: Number(value),
    };
  }).reverse();
}

export function transformRumSessionTimeResponse(response: RumSessionTimeResponse) {
  const totalSeconds = Number(response.seconds);
  const minutes = Math.floor(totalSeconds / 60);
  const seconds = Math.floor(totalSeconds % 60);

  const paddedMinutes = String(minutes).padStart(2, '0');
  const paddedSeconds = String(seconds).padStart(2, '0');

  return { minutes: paddedMinutes, seconds: paddedSeconds };
}